<?php
/*--------------------------------------------------------------------------------------------------
    ResponsiveGroupOption.php 2022-08-05
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2022 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------------------------------------*/

namespace Gambio\StyleEdit\Core\Components\ResponsiveGroup\Entities;

use Gambio\StyleEdit\Core\Options\Entities\AbstractComponentGroupOption;
use Gambio\StyleEdit\Core\Options\Entities\AbstractComponentOption;
use Gambio\StyleEdit\Core\SingletonPrototype;
use Gambio\StyleEdit\Core\Components\Option\Entities\Option;
use stdClass;

/**
 * Class PaddingGroupOption
 * @package Gambio\StyleEdit\Core\Components\PaddingGroup\Entities
 */
class ResponsiveGroupOption extends AbstractComponentGroupOption
{
    /**
     * @var Option
     */
    protected $hiddenXs;
    
    /**
     * @var Option
     */
    protected $hiddenSm;
    
    /**
     * @var Option
     */
    protected $hiddenMd;
    
    /**
     * @var Option
     */
    protected $hiddenLg;
    
    
    /**
     * PaddingGroupOption constructor.
     */
    public function __construct()
    {
        parent::__construct();
        
        $this->hiddenXs = SingletonPrototype::instance()->get('Option');
        $this->hiddenSm = SingletonPrototype::instance()->get('Option');
        $this->hiddenMd = SingletonPrototype::instance()->get('Option');
        $this->hiddenLg = SingletonPrototype::instance()->get('Option');
    }
    
    
    public function __clone()
    {
        parent::__clone(); // TODO: Change the autogenerated stub
        $this->hiddenXs = clone $this->hiddenXs;
        $this->hiddenSm = clone $this->hiddenSm;
        $this->hiddenMd = clone $this->hiddenMd;
        $this->hiddenLg = clone $this->hiddenLg;
    }
    
    
    /**
     * @return mixed
     */
    public function hiddenXs()
    {
        return $this->hiddenXs;
    }
    
    
    /**
     * @return mixed
     */
    public function hiddenSm()
    {
        return $this->hiddenSm;
    }
    
    
    /**
     * @return mixed
     */
    public function hiddenMd()
    {
        return $this->hiddenMd;
    }
    
    
    /**
     * @return mixed
     */
    public function hiddenLg()
    {
        return $this->hiddenLg;
    }
    
    
    /**
     * Specify data which should be serialized to JSON
     * @link  https://php.net/manual/en/jsonserializable.jsonserialize.php
     * @return mixed data which can be serialized by <b>json_encode</b>,
     * which is a value of any type other than a resource.
     * @since 5.4.0
     */
    #[\ReturnTypeWillChange]
    public function jsonSerialize()
    {
        $result        = new stdClass();
        $result->id    = $this->id();
        $result->type  = 'responsive';
        $result->label = $this->label();
    
        if ($this->pageNamespace()) {
            $result->pageNamespace = $this->pageNamespace();
        }
        
        $result->items = [
            'hiddenXs' => (bool)$this->hiddenXs()->value(),
            'hiddenSm' => (bool)$this->hiddenSm()->value(),
            'hiddenMd' => (bool)$this->hiddenMd()->value(),
            'hiddenLg' => (bool)$this->hiddenLg()->value()
        ];
        
        $result->default = [
            'hiddenXs' => (bool)$this->hiddenXs()->defaultValue(),
            'hiddenSm' => (bool)$this->hiddenSm()->defaultValue(),
            'hiddenMd' => (bool)$this->hiddenMd()->defaultValue(),
            'hiddenLg' => (bool)$this->hiddenLg()->defaultValue()
        ];
        
        return $result;
    }
    
    
    /**
     * @param $value
     *
     * @return boolean
     */
    protected function isValid($value): bool
    {
        return true;
    }
    
    
    /**
     * @param $value
     *
     * @return mixed
     */
    protected function parseValue($value)
    {
        return $value;
    }
    
    
    /**
     * @param $object
     *
     * @throws \Exception
     */
    public function initializeFromJsonObject($object): void
    {
        if (isset($object->id)) {
            $this->id = $object->id;
        }
        
        if (isset($object->label)) {
            $this->label = $object->label;
        }

        if (isset($object->pageNamespace)) {
            $this->pageNamespace = $object->pageNamespace;
        }
        
        $settings = ['id' => $this->id() . '-hidden-xs'];
        if (isset($object->default->hiddenXs)) {
            $settings['default'] = $object->default->hiddenXs;
        }
        if (isset($object->items->hiddenXs)) {
            $settings['value'] = $object->items->hiddenXs;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->hiddenXs()
            ->withConfigurationRepository($this->configurationRepository())
            ->initializeFromJsonObject((object)$settings);
        
        $settings = ['id' => $this->id() . '-hidden-sm'];
        if (isset($object->default->hiddenSm)) {
            $settings['default'] = $object->default->hiddenSm;
        }
        if (isset($object->items->hiddenSm)) {
            $settings['value'] = $object->items->hiddenSm;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->hiddenSm->initializeFromJsonObject((object)$settings);
        
        $settings = ['id' => $this->id() . '-hidden-md'];
        if (isset($object->default->hiddenMd)) {
            $settings['default'] = $object->default->hiddenMd;
        }
        if (isset($object->items->hiddenMd)) {
            $settings['value'] = $object->items->hiddenMd;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->hiddenMd()
            ->withConfigurationRepository($this->configurationRepository())
            ->initializeFromJsonObject((object)$settings);
        
        $settings = ['id' => $this->id() . '-hidden-lg'];
        if (isset($object->default->hiddenLg)) {
            $settings['default'] = $object->default->hiddenLg;
        }
        if (isset($object->items->hiddenLg)) {
            $settings['value'] = $object->items->hiddenLg;
        }
        if (isset($object->group)) {
            $settings['group'] = $object->group;
        }
        if (isset($object->for)) {
            $settings['for'] = $object->for;
        }
        $this->hiddenLg()
            ->withConfigurationRepository($this->configurationRepository())
            ->initializeFromJsonObject((object)$settings);
    }
    
    
    /**
     * @return mixed
     */
    public function getGroupOptions()
    {
        return [
            $this->hiddenLg(),
            $this->hiddenMd(),
            $this->hiddenSm(),
            $this->hiddenXs()
        ];
    }
    
    
    /**
     * @return string
     */
    public function type(): ?string
    {
        return 'responsive';
    }
}